<x-app-layout title="New Purchase Order">

    <div class="flex flex-col md:flex-row md:items-center md:justify-between gap-4">
        <div>
            <p class="text-sm text-textSecondary uppercase tracking-[0.3em]">Purchases</p>
            <h1 class="text-3xl font-semibold mt-2">Create Purchase Order</h1>
        </div>
        <a href="{{ route('purchases.index') }}" class="text-primary hover:underline text-sm"><- Back to list</a>
    </div>

    @php
        $variationMeta = $variations->map(function ($variation) {
            return [
                'id' => $variation->id,
                'product_id' => $variation->product_id,
                'product_name' => optional($variation->product)->name,
                'sku' => $variation->sku,
                'color' => $variation->color,
                'storage' => $variation->storage,
                'serial_number' => $variation->serial_number,
            ];
        })->values();


        $phoneSerialMap = $variations->mapWithKeys(function ($variation) {
            if (! $variation->serial_number) {
                return [];
            }
            return [$variation->serial_number => ['id' => $variation->id]];
        });


        $initialStep = 1;
        if ($errors->any()) {
            $initialStep = $errors->hasAny(['invoice_number', 'supplier_id', 'purchase_date', 'purchase_type']) ? 1 : 2;
        }

        $purchaseFormPayload = [
            'purchaseCode' => old('purchase_code', $purchaseCode),
            'sInvoiceNumber' => old('s_invoice_number', $sInvoiceNumber),
            'invoiceNumber' => old('invoice_number'),
            'purchaseDate' => old('purchase_date', now()->toDateString()),
            'initialStep' => $initialStep,
            'variations' => $variationMeta,
            'phoneSerialMap' => $phoneSerialMap,
            'computerOptions' => $computerOptions,
            'oldItems' => old('items', []),
            'defaults' => [
                'tax_amount' => old('tax_amount', 0),
                'other_charges' => old('other_charges', 0),
            ],
            'oldSupplier' => old('supplier_id'),
            'purchaseType' => old('purchase_type', 'phone'),
        ];
    @endphp

    <form action="{{ route('purchases.store') }}" method="POST" class="mt-8 space-y-6" x-data='purchaseForm(@json($purchaseFormPayload))' novalidate>
        @csrf

        <div class="rounded-3xl border border-borderColor/60 bg-white/70 px-6 py-4">
            <div class="flex flex-col md:flex-row md:items-center md:justify-between gap-4 text-sm">
                <div class="flex items-center gap-3">
                    <span class="px-3 py-1 rounded-full text-xs font-semibold"
                          :class="step >= 1 ? 'bg-primary text-white' : 'bg-slate-100 text-textSecondary'">Step 1</span>
                    <span class="font-medium text-textPrimary">Purchase Info</span>
                </div>
                <div class="flex items-center gap-3">
                    <span class="px-3 py-1 rounded-full text-xs font-semibold"
                          :class="step >= 2 ? 'bg-primary text-white' : 'bg-slate-100 text-textSecondary'">Step 2</span>
                    <span class="font-medium text-textPrimary">Product Details</span>
                </div>
                <div class="flex items-center gap-3">
                    <span class="px-3 py-1 rounded-full text-xs font-semibold"
                          :class="step >= 3 ? 'bg-primary text-white' : 'bg-slate-100 text-textSecondary'">Step 3</span>
                    <span class="font-medium text-textPrimary">Summary & Payment</span>
                </div>
            </div>
        </div>

        <div class="space-y-6">
            <div class="glass-card rounded-3xl p-6 space-y-5" x-show="step === 1">
                <div class="space-y-1">
                    <h2 class="text-lg font-semibold">Information</h2>
                    <p class="text-sm text-textSecondary">Capture supplier orders and track inbound stock.</p>
                </div>

                <div class="grid md:grid-cols-3 gap-4">
                    <div>
                        <label class="text-sm text-textSecondary font-medium mb-1 block">Purchase ID</label>
                        <input type="text"
                               value="{{ old('purchase_code', $purchaseCode) }}"
                               x-bind:value="purchaseCode"
                               readonly
                               class="w-full rounded-2xl border border-white/60 bg-slate-50 px-4 py-3 text-textSecondary">
                        <input type="hidden" name="purchase_code" value="{{ old('purchase_code', $purchaseCode) }}" x-bind:value="purchaseCode">
                        <p class="text-xs text-textSecondary mt-1">Auto generated from current datetime.</p>
                    </div>
                    <div>
                        <label class="text-sm text-textSecondary font-medium mb-1 block">Invoice Number</label>
                        <input type="text" name="invoice_number" value="{{ old('invoice_number') }}"
                               x-model="invoiceNumber"
                               class="w-full rounded-2xl border border-white/60 bg-white/80 px-4 py-3 focus:ring-primary/40 focus:border-primary/50"
                               placeholder="Supplier invoice"
                               required>
                    </div>
                    <div>
                        <label class="text-sm text-textSecondary font-medium mb-1 block">S-Invoice</label>
                        <input type="text"
                               value="{{ old('s_invoice_number', $sInvoiceNumber) }}"
                               x-bind:value="sInvoiceNumber"
                               readonly
                               class="w-full rounded-2xl border border-white/60 bg-slate-50 px-4 py-3 text-textSecondary">
                        <input type="hidden" name="s_invoice_number" value="{{ old('s_invoice_number', $sInvoiceNumber) }}" x-bind:value="sInvoiceNumber">
                        <p class="text-xs text-textSecondary mt-1">Format: YearMonthDayHourMinute.</p>
                    </div>
                </div>

                <div class="grid md:grid-cols-3 gap-4">
                    <div>
                        <label class="text-sm text-textSecondary font-medium mb-1 block">Supplier</label>
                        <select name="supplier_id"
                                x-model="supplierId"
                                class="w-full rounded-2xl border border-white/60 bg-white/80 px-4 py-3 focus:ring-primary/40 focus:border-primary/50"
                                required>
                            <option value="">Select supplier</option>
                            @foreach($suppliers as $supplier)
                                <option value="{{ $supplier->id }}" @selected(old('supplier_id') == $supplier->id)>{{ $supplier->name }}</option>
                            @endforeach
                        </select>
                        @if($suppliers->isEmpty())
                            <p class="text-xs text-red-500 mt-2">Add a supplier first to create purchase orders.</p>
                        @endif
                        <p class="text-xs text-textSecondary mt-1" x-show="supplierId">
                            Supplier ID: <span class="font-semibold" x-text="supplierId"></span>
                        </p>
                    </div>
                    <div>
                        <label class="text-sm text-textSecondary font-medium mb-1 block">Purchase Date</label>
                        <input type="date" name="purchase_date" value="{{ now()->toDateString() }}"
                               x-model="purchaseDate"
                               class="w-full rounded-2xl border border-white/60 bg-white/80 px-4 py-3 focus:ring-primary/40 focus:border-primary/50" required>
                    </div>
                    <div>
                        <label class="text-sm text-textSecondary font-medium mb-1 block">Purchase Type</label>
                        <select name="purchase_type"
                                x-model="purchaseType"
                                @change="resetItemsForType"
                                class="w-full rounded-2xl border border-white/60 bg-white/80 px-4 py-3 focus:ring-primary/40 focus:border-primary/50"
                                required>
                            <option value="">Select type</option>
                            <option value="computer" @selected(old('purchase_type') === 'computer')>Computer</option>
                            <option value="phone" @selected(old('purchase_type') === 'phone')>Phone</option>
                        </select>
                    </div>
                </div>
            </div>

            <div class="glass-card rounded-3xl p-6 space-y-4" x-show="step === 2">
                <div class="flex items-center justify-between">
                    <div>
                        <h2 class="text-lg font-semibold">Product Details</h2>
                        <p class="text-sm text-textSecondary">Add each SKU, quantity, and cost.</p>
                    </div>
                    <button type="button" class="px-4 py-2 rounded-2xl bg-primary text-white text-sm shadow shadow-primary/30" @click="addItem">
                        + Add Item
                    </button>
                </div>

                <template x-for="(item, index) in items" :key="item.uid">
                    <div class="rounded-2xl border border-white/60 bg-white/75 p-4 space-y-4">
                        <div class="flex items-start justify-between gap-4">
                            <div class="flex-1 space-y-4">
                                <template x-if="purchaseType === 'phone'">
                                    <div class="space-y-3">
                                        <div class="grid md:grid-cols-3 gap-3">
                                            <div>
                                                <label class="text-sm text-textSecondary font-medium mb-1 block">Product / SKU</label>
                                                <select x-model="item.variation_id"
                                                        :name="`items[${index}][variation_id]`"
                                                        @change="syncSelection(item)"
                                                        class="w-full rounded-2xl border border-borderColor/60 px-4 py-3 focus:ring-primary/40 focus:border-primary/50"
                                                        :required="step === 2 && purchaseType === 'phone'">
                                                    <option value="">Select SKU</option>
                                                    @foreach($variations as $variation)
                                                        <option value="{{ $variation->id }}">
                                                            {{ optional($variation->product)->name ?? 'Product #'.$variation->product_id }} - SKU {{ $variation->sku }}
                                                            @if($variation->color || $variation->storage)
                                                                ({{ collect([$variation->color, $variation->storage])->filter()->join(' / ') }})
                                                            @endif
                                                        </option>
                                                    @endforeach
                                                </select>
                                                <p class="text-xs text-textSecondary mt-1" x-show="item.product_id">
                                                    Product ID:
                                                    <span class="font-semibold" x-text="item.product_id"></span>
                                                </p>
                                            </div>
                                            <div>
                                                <label class="text-sm text-textSecondary font-medium mb-1 block">Quantity</label>
                                                <input type="number" min="1"
                                                       x-model.number="item.qty"
                                                       @change="syncSerials(item)"
                                                       :name="`items[${index}][qty]`"
                                                       class="w-full rounded-2xl border border-borderColor/60 px-4 py-3 focus:ring-primary/40 focus:border-primary/50"
                                                       :required="step === 2">
                                            </div>
                                            <div class="rounded-2xl border border-white/60 bg-white/70 px-4 py-3" x-show="item.details">
                                                <p class="text-xs uppercase tracking-[0.2em] text-textSecondary">SKU</p>
                                                <p class="font-semibold" x-text="item.details?.sku || '-'"></p>
                                                <p class="text-xs uppercase tracking-[0.2em] text-textSecondary mt-2">Color / Storage</p>
                                                <p class="font-medium" x-text="formatVariant(item.details)"></p>
                                            </div>
                                        </div>
                                        <div class="space-y-2">
                                            <div class="flex items-center justify-between">
                                                <label class="text-sm text-textSecondary font-medium">Serial Numbers (SN)</label>
                                                <button type="button"
                                                        class="text-xs text-primary hover:underline"
                                                        @click="addSerial(item)">
                                                    + Add SN
                                                </button>
                                            </div>
                                            <template x-for="(serial, serialIndex) in item.serial_numbers" :key="`phone-${item.uid}-${serialIndex}`">
                                                <div class="flex items-center gap-2">
                                                    <input type="text"
                                                           x-model="item.serial_numbers[serialIndex]"
                                                           list="phone-serials"
                                                           @change="handleSerialChange(item, serialIndex)"
                                                           :name="`items[${index}][serial_numbers][${serialIndex}]`"
                                                           class="w-full rounded-2xl border border-borderColor/60 px-4 py-3 focus:ring-primary/40 focus:border-primary/50"
                                                           placeholder="Enter serial number"
                                                           :required="step === 2">
                                                    <button type="button"
                                                            class="text-xs text-red-500 hover:underline"
                                                            @click="removeSerial(item, serialIndex)"
                                                            x-show="item.serial_numbers.length > 1">
                                                        Remove
                                                    </button>
                                                </div>
                                            </template>
                                            <p class="text-xs text-red-500" x-show="serialsMismatch(item)">
                                                Serial number count must match quantity.
                                            </p>
                                        </div>
                                    </div>
                                </template>

                                <template x-if="purchaseType === 'computer'">
                                    <div class="space-y-4">
                                        <div class="grid md:grid-cols-4 gap-3">
                                            <div class="md:col-span-2">
                                                <label class="text-sm text-textSecondary font-medium mb-1 block">Product ID</label>
                                                <div class="relative">
                                                    <input type="text"
                                                           x-model="item.product_code_query"
                                                           @input.debounce.300ms="searchComputers(item)"
                                                           @focus="searchComputers(item)"
                                                           class="w-full rounded-2xl border border-borderColor/60 px-4 py-3 focus:ring-primary/40 focus:border-primary/50"
                                                           placeholder="Search Product ID"
                                                           :required="step === 2 && purchaseType === 'computer'">
                                                    <input type="hidden" :name="`items[${index}][computer_product_id]`" x-bind:value="item.computer_product_id">
                                                    <input type="hidden" :name="`items[${index}][product_code]`" x-bind:value="item.product_code">
                                                    <div class="absolute z-10 mt-2 w-full rounded-2xl border border-borderColor/60 bg-white shadow-lg max-h-56 overflow-auto"
                                                         x-show="item.searching || (item.searchResults && item.searchResults.length)">
                                                        <div class="px-4 py-3 text-sm text-textSecondary" x-show="item.searching">Searching inventory...</div>
                                                        <template x-for="result in (item.searchResults || [])" :key="result.id">
                                                            <button type="button"
                                                                    class="w-full text-left px-4 py-2 hover:bg-slate-50"
                                                                    @click="selectComputer(item, result)">
                                                                <p class="font-semibold text-textPrimary" x-text="result.product_code"></p>
                                                                <p class="text-xs text-textSecondary" x-text="result.name"></p>
                                                            </button>
                                                        </template>
                                                    </div>
                                                </div>
                                                <div class="flex items-center justify-between mt-2">
                                                    <p class="text-xs text-textSecondary" x-show="!item.searching && item.product_code_query && (!item.searchResults || !item.searchResults.length)">No matching Product ID.</p>
                                                    <button type="button"
                                                            class="text-xs text-primary hover:underline"
                                                            x-show="item.product_code_query && (!item.searchResults || !item.searchResults.length) && !item.searching"
                                                            @click="openComputerModal(item)">
                                                        + Add New Computer
                                                    </button>
                                                </div>
                                                <p class="text-xs text-red-500 mt-1" x-show="item.searchError" x-text="item.searchError"></p>
                                            </div>
                                            <div>
                                                <label class="text-sm text-textSecondary font-medium mb-1 block">Quantity</label>
                                                <input type="number" min="1"
                                                       x-model.number="item.qty"
                                                       :name="`items[${index}][qty]`"
                                                       @change="syncSerials(item)"
                                                       class="w-full rounded-2xl border border-borderColor/60 px-4 py-3 focus:ring-primary/40 focus:border-primary/50"
                                                       :required="step === 2">
                                            </div>
                                            <div>
                                                <label class="text-sm text-textSecondary font-medium mb-1 block">Serial Number</label>
                                                <div class="space-y-2">
                                                    <div class="flex items-center justify-between">
                                                        <span class="text-xs text-textSecondary">Serial Numbers (SN)</span>
                                                        <button type="button"
                                                                class="text-xs text-primary hover:underline"
                                                                @click="addSerial(item)">
                                                            + Add SN
                                                        </button>
                                                    </div>
                                                    <template x-for="(serial, serialIndex) in item.serial_numbers" :key="`computer-${item.uid}-${serialIndex}`">
                                                        <div class="flex items-center gap-2">
                                                            <input type="text"
                                                                   x-model="item.serial_numbers[serialIndex]"
                                                                   @change="handleSerialChange(item, serialIndex)"
                                                                   :name="`items[${index}][serial_numbers][${serialIndex}]`"
                                                                   class="w-full rounded-2xl border border-borderColor/60 px-4 py-3 focus:ring-primary/40 focus:border-primary/50"
                                                                   placeholder="Unique per device"
                                                                   :required="step === 2">
                                                            <button type="button"
                                                                    class="text-xs text-red-500 hover:underline"
                                                                    @click="removeSerial(item, serialIndex)"
                                                                    x-show="item.serial_numbers.length > 1">
                                                                Remove
                                                            </button>
                                                        </div>
                                                    </template>
                                                    <p class="text-xs text-red-500" x-show="serialsMismatch(item)">
                                                        Serial number count must match quantity.
                                                    </p>
                                                </div>
                                            </div>
                                        </div>
                                        <div class="grid md:grid-cols-3 gap-3">
                                            <div>
                                                <label class="text-sm text-textSecondary font-medium mb-1 block">CodeName</label>
                                                <input type="text" x-model="item.code_name" readonly
                                                       class="w-full rounded-2xl border border-white/60 bg-slate-50 px-4 py-3 text-textSecondary">
                                            </div>
                                            <div>
                                                <label class="text-sm text-textSecondary font-medium mb-1 block">Product Name</label>
                                                <input type="text" x-model="item.name" readonly
                                                       class="w-full rounded-2xl border border-white/60 bg-slate-50 px-4 py-3 text-textSecondary">
                                            </div>
                                            <div>
                                                <label class="text-sm text-textSecondary font-medium mb-1 block">Display</label>
                                                <input type="text" x-model="item.display" readonly
                                                       class="w-full rounded-2xl border border-white/60 bg-slate-50 px-4 py-3 text-textSecondary">
                                            </div>
                                        </div>
                                        <div class="grid md:grid-cols-3 gap-3">
                                            <div>
                                                <label class="text-sm text-textSecondary font-medium mb-1 block">Internal ID (A13_1)</label>
                                                <input type="text" x-model="item.internal_id" readonly
                                                       class="w-full rounded-2xl border border-white/60 bg-slate-50 px-4 py-3 text-textSecondary">
                                            </div>
                                        </div>
                                        <div class="grid md:grid-cols-3 gap-3">
                                            <div>
                                                <label class="text-sm text-textSecondary font-medium mb-1 block">CPU</label>
                                                <input type="text" x-model="item.cpu" readonly
                                                       class="w-full rounded-2xl border border-white/60 bg-slate-50 px-4 py-3 text-textSecondary">
                                            </div>
                                            <div>
                                                <label class="text-sm text-textSecondary font-medium mb-1 block">Capacity</label>
                                                <input type="text" x-model="item.capacity" readonly
                                                       class="w-full rounded-2xl border border-white/60 bg-slate-50 px-4 py-3 text-textSecondary">
                                            </div>
                                            <div>
                                                <label class="text-sm text-textSecondary font-medium mb-1 block">RAM</label>
                                                <input type="text" x-model="item.ram" readonly
                                                       class="w-full rounded-2xl border border-white/60 bg-slate-50 px-4 py-3 text-textSecondary">
                                            </div>
                                        </div>
                                        <div class="grid md:grid-cols-3 gap-3">
                                            <div>
                                                <label class="text-sm text-textSecondary font-medium mb-1 block">Storage</label>
                                                <input type="text" x-model="item.storage" readonly
                                                       class="w-full rounded-2xl border border-white/60 bg-slate-50 px-4 py-3 text-textSecondary">
                                            </div>
                                            <div>
                                                <label class="text-sm text-textSecondary font-medium mb-1 block">ColorCode</label>
                                                <input type="text" x-model="item.color_code" readonly
                                                       class="w-full rounded-2xl border border-white/60 bg-slate-50 px-4 py-3 text-textSecondary">
                                            </div>
                                            <div>
                                                <label class="text-sm text-textSecondary font-medium mb-1 block">Color</label>
                                                <input type="text" x-model="item.color" readonly
                                                       class="w-full rounded-2xl border border-white/60 bg-slate-50 px-4 py-3 text-textSecondary">
                                            </div>
                                        </div>
                                        <div class="grid md:grid-cols-3 gap-3">
                                            <div>
                                                <label class="text-sm text-textSecondary font-medium mb-1 block">CountryCode</label>
                                                <input type="text" x-model="item.country_code" readonly
                                                       class="w-full rounded-2xl border border-white/60 bg-slate-50 px-4 py-3 text-textSecondary">
                                            </div>
                                            <div>
                                                <label class="text-sm text-textSecondary font-medium mb-1 block">Country</label>
                                                <input type="text" x-model="item.country" readonly
                                                       class="w-full rounded-2xl border border-white/60 bg-slate-50 px-4 py-3 text-textSecondary">
                                            </div>
                                            <div>
                                                <label class="text-sm text-textSecondary font-medium mb-1 block">Condition</label>
                                                <input type="text" x-model="item.condition" readonly
                                                       class="w-full rounded-2xl border border-white/60 bg-slate-50 px-4 py-3 text-textSecondary">
                                            </div>
                                        </div>
                                        <div class="grid md:grid-cols-3 gap-3">
                                            <div>
                                                <label class="text-sm text-textSecondary font-medium mb-1 block">Activate</label>
                                                <input type="text" x-model="item.activate" readonly
                                                       class="w-full rounded-2xl border border-white/60 bg-slate-50 px-4 py-3 text-textSecondary">
                                            </div>
                                            <div class="rounded-2xl border border-white/60 bg-white/70 px-4 py-3">
                                                <p class="text-xs uppercase tracking-[0.2em] text-textSecondary">Snapshot</p>
                                                <p class="font-semibold" x-text="item.name || 'Product details'"></p>
                                                <p class="text-sm text-textSecondary" x-text="buildComputerSummary(item)"></p>
                                            </div>
                                        </div>
                                    </div>
                                </template>
                            </div>
                            <div class="flex items-center gap-3">
                                <button type="button" class="text-sm text-red-500 hover:underline" @click="removeItem(index)" x-show="items.length > 1">
                                    Remove
                                </button>
                            </div>
                        </div>

                        <div class="grid md:grid-cols-2 gap-4">
                            <div>
                                <label class="text-sm text-textSecondary font-medium mb-1 block">Cost Price</label>
                                <input type="number" min="0" step="0.01"
                                       x-model.number="item.cost_price"
                                       :name="`items[${index}][cost_price]`"
                                       class="w-full rounded-2xl border border-borderColor/60 px-4 py-3 focus:ring-primary/40 focus:border-primary/50"
                                       :required="step === 2">
                            </div>
                            <div class="md:col-span-1">
                                <label class="text-sm text-textSecondary font-medium mb-1 block">Product Snapshot</label>
                                <div class="rounded-2xl border border-white/60 bg-white/70 px-4 py-3 space-y-1" x-show="itemSummary(item)">
                                    <p class="font-semibold" x-text="summaryTitle(item)"></p>
                                    <p class="text-sm text-textSecondary" x-text="itemSummary(item)"></p>
                                </div>
                                <div class="rounded-2xl border border-white/60 bg-white/60 px-4 py-3 text-sm text-textSecondary" x-show="!itemSummary(item)">
                                    Complete the fields above to view details.
                                </div>
                            </div>
                        </div>
                    </div>
                </template>

                @if ($errors->any())
                    <div class="rounded-2xl bg-red-50 border border-red-200 text-sm text-red-600 px-4 py-3">
                        <ul class="list-disc list-inside space-y-1">
                            @foreach ($errors->all() as $error)
                                <li>{{ $error }}</li>
                            @endforeach
                        </ul>
                    </div>
                @endif
            </div>

            <div class="glass-card rounded-3xl p-6 space-y-5" x-show="step === 3">
                <h2 class="text-lg font-semibold">Summary</h2>
                <div class="flex items-center justify-between">
                    <span class="text-textSecondary">Line Items</span>
                    <span class="font-semibold" x-text="items.length"></span>
                </div>
                <div class="space-y-3">
                    <div class="space-y-2">
                        <template x-for="(item, index) in items" :key="`summary-${item.uid}`">
                            <div class="rounded-2xl border border-white/60 bg-white/70 px-4 py-3">
                                <div class="flex items-center justify-between">
                                    <p class="font-semibold" x-text="summaryTitle(item)"></p>
                                    <p class="text-sm text-textSecondary" x-text="`Qty ${item.qty}`"></p>
                                </div>
                                <div class="flex items-center justify-between text-sm text-textSecondary">
                                    <span x-text="`Cost ${formatCurrency(item.cost_price)}`"></span>
                                    <span x-text="`Subtotal ${formatCurrency(item.qty * item.cost_price)}`"></span>
                                </div>
                            </div>
                        </template>
                    </div>
                    <div>
                        <label class="text-sm text-textSecondary font-medium mb-1 block">Tax</label>
                        <input type="number" min="0" step="0.01"
                               name="tax_amount"
                               x-model.number="taxAmount"
                               value="{{ old('tax_amount', 0) }}"
                               class="w-full rounded-2xl border border-white/60 bg-white/80 px-4 py-3 focus:ring-primary/40 focus:border-primary/50"
                               placeholder="0.00">
                    </div>
                    <div>
                        <label class="text-sm text-textSecondary font-medium mb-1 block">Other Charges</label>
                        <input type="number" min="0" step="0.01"
                               name="other_charges"
                               x-model.number="otherCharges"
                               value="{{ old('other_charges', 0) }}"
                               class="w-full rounded-2xl border border-white/60 bg-white/80 px-4 py-3 focus:ring-primary/40 focus:border-primary/50"
                               placeholder="0.00">
                    </div>
                    <div class="rounded-2xl bg-white/70 border border-white/60 px-4 py-3">
                        <p class="text-sm text-textSecondary">Total Amount</p>
                        <p class="text-2xl font-semibold" x-text="formatCurrency(totalAmount)"></p>
                    </div>
                    <div class="rounded-2xl bg-primary/10 border border-primary/30 px-4 py-3">
                        <p class="text-sm text-textSecondary">Total Amount Due</p>
                        <p class="text-2xl font-semibold" x-text="formatCurrency(totalDue)"></p>
                    </div>
                    <input type="hidden" name="total_due" value="{{ old('total_due', 0) }}" x-bind:value="totalDue.toFixed(2)">
                </div>
            </div>

            <div class="flex flex-col md:flex-row md:items-center md:justify-between gap-3">
                <button type="button"
                        class="px-6 py-3 rounded-2xl border border-borderColor/70 text-textSecondary hover:bg-slate-50"
                        x-show="step > 1"
                        @click="prevStep">
                    Back
                </button>
                <div class="flex items-center justify-end gap-3">
                    <button type="button"
                            class="px-6 py-3 rounded-2xl bg-slate-100 text-textSecondary hover:bg-slate-200"
                            x-show="step < 3"
                            :disabled="!canProceed"
                            @click="nextStep">
                        Next
                    </button>
                    <button type="submit"
                            class="px-8 py-3 rounded-2xl bg-gradient-to-r from-primary to-primaryDark text-white shadow-lg shadow-primary/40"
                            x-show="step === 3">
                        Save Purchase Order
                    </button>
                </div>
            </div>

            <div class="fixed inset-0 z-50 flex items-center justify-center px-4 py-6 bg-black/40"
                 x-show="showComputerModal"
                 x-cloak>
                <div class="bg-white rounded-3xl shadow-xl w-full max-w-3xl p-6" @click.outside="showComputerModal = false">
                    <div class="flex items-center justify-between">
                        <h3 class="text-lg font-semibold">Add New Computer Product</h3>
                        <button type="button" class="text-sm text-textSecondary hover:text-primary" @click="showComputerModal = false">Close</button>
                    </div>
                    <div class="mt-4 flex items-center gap-3 text-sm">
                        <button type="button"
                                class="px-3 py-1 rounded-full border"
                                :class="computerModalTab === 'create' ? 'bg-primary text-white border-primary' : 'border-borderColor/60 text-textSecondary'"
                                @click="computerModalTab = 'create'">
                            Create
                        </button>
                        <button type="button"
                                class="px-3 py-1 rounded-full border"
                                :class="computerModalTab === 'import' ? 'bg-primary text-white border-primary' : 'border-borderColor/60 text-textSecondary'"
                                @click="computerModalTab = 'import'">
                            Import
                        </button>
                    </div>

                    <div class="mt-6" x-show="computerModalTab === 'create'">
                        <div class="grid md:grid-cols-2 gap-4">
                            <div>
                                <label class="text-sm text-textSecondary font-medium mb-1 block">CodeName</label>
                                <select x-model="computerForm.code_name"
                                        class="w-full rounded-2xl border border-borderColor/60 px-4 py-3 focus:ring-primary/40 focus:border-primary/50">
                                    <option value="">Select CodeName</option>
                                    <template x-for="option in (computerOptions.code_names || [])" :key="option">
                                        <option :value="option" x-text="option"></option>
                                    </template>
                                </select>
                            </div>
                            <div>
                                <label class="text-sm text-textSecondary font-medium mb-1 block">Product Name</label>
                                <select x-model="computerForm.name"
                                        class="w-full rounded-2xl border border-borderColor/60 px-4 py-3 focus:ring-primary/40 focus:border-primary/50">
                                    <option value="">Select Product Name</option>
                                    <template x-for="option in (computerOptions.product_names || [])" :key="option">
                                        <option :value="option" x-text="option"></option>
                                    </template>
                                </select>
                            </div>
                        </div>

                        <div class="grid md:grid-cols-2 gap-4 mt-4">
                            <div>
                                <label class="text-sm text-textSecondary font-medium mb-1 block">Display</label>
                                <select x-model="computerForm.display"
                                        class="w-full rounded-2xl border border-borderColor/60 px-4 py-3 focus:ring-primary/40 focus:border-primary/50">
                                    <option value="">Select Display</option>
                                    <template x-for="option in (computerOptions.displays || [])" :key="option">
                                        <option :value="option" x-text="option"></option>
                                    </template>
                                </select>
                            </div>
                            <div>
                                <label class="text-sm text-textSecondary font-medium mb-1 block">CPU</label>
                                <select x-model="computerForm.cpu"
                                        class="w-full rounded-2xl border border-borderColor/60 px-4 py-3 focus:ring-primary/40 focus:border-primary/50">
                                    <option value="">Select CPU</option>
                                    <template x-for="option in (computerOptions.cpus || [])" :key="option">
                                        <option :value="option" x-text="option"></option>
                                    </template>
                                </select>
                            </div>
                        </div>

                        <div class="grid md:grid-cols-2 gap-4 mt-4">
                            <div>
                                <label class="text-sm text-textSecondary font-medium mb-1 block">Capacity</label>
                                <select x-model="computerForm.capacity"
                                        class="w-full rounded-2xl border border-borderColor/60 px-4 py-3 focus:ring-primary/40 focus:border-primary/50">
                                    <option value="">Select Capacity</option>
                                    <template x-for="option in (computerOptions.capacities || [])" :key="option">
                                        <option :value="option" x-text="option"></option>
                                    </template>
                                </select>
                            </div>
                            <div>
                                <label class="text-sm text-textSecondary font-medium mb-1 block">RAM</label>
                                <select x-model="computerForm.ram"
                                        class="w-full rounded-2xl border border-borderColor/60 px-4 py-3 focus:ring-primary/40 focus:border-primary/50">
                                    <option value="">Select RAM</option>
                                    <template x-for="option in (computerOptions.rams || [])" :key="option">
                                        <option :value="option" x-text="option"></option>
                                    </template>
                                </select>
                            </div>
                        </div>

                        <div class="grid md:grid-cols-2 gap-4 mt-4">
                            <div>
                                <label class="text-sm text-textSecondary font-medium mb-1 block">Storage</label>
                                <select x-model="computerForm.storage"
                                        class="w-full rounded-2xl border border-borderColor/60 px-4 py-3 focus:ring-primary/40 focus:border-primary/50">
                                    <option value="">Select Storage</option>
                                    <template x-for="option in (computerOptions.storages || [])" :key="option">
                                        <option :value="option" x-text="option"></option>
                                    </template>
                                </select>
                            </div>
                            <div>
                                <label class="text-sm text-textSecondary font-medium mb-1 block">Color Code</label>
                                <select x-model="computerForm.color_code"
                                        class="w-full rounded-2xl border border-borderColor/60 px-4 py-3 focus:ring-primary/40 focus:border-primary/50">
                                    <option value="">Select Color Code</option>
                                    <template x-for="option in (computerOptions.color_codes || [])" :key="option">
                                        <option :value="option" x-text="option"></option>
                                    </template>
                                </select>
                            </div>
                        </div>

                        <div class="grid md:grid-cols-2 gap-4 mt-4">
                            <div>
                                <label class="text-sm text-textSecondary font-medium mb-1 block">Color</label>
                                <select x-model="computerForm.color"
                                        class="w-full rounded-2xl border border-borderColor/60 px-4 py-3 focus:ring-primary/40 focus:border-primary/50">
                                    <option value="">Select Color</option>
                                    <template x-for="option in (computerOptions.colors || [])" :key="option">
                                        <option :value="option" x-text="option"></option>
                                    </template>
                                </select>
                            </div>
                            <div>
                                <label class="text-sm text-textSecondary font-medium mb-1 block">Country Code</label>
                                <select x-model="computerForm.country_code"
                                        class="w-full rounded-2xl border border-borderColor/60 px-4 py-3 focus:ring-primary/40 focus:border-primary/50">
                                    <option value="">Select Country Code</option>
                                    <template x-for="option in (computerOptions.country_codes || [])" :key="option">
                                        <option :value="option" x-text="option"></option>
                                    </template>
                                </select>
                            </div>
                        </div>

                        <div class="grid md:grid-cols-2 gap-4 mt-4">
                            <div>
                                <label class="text-sm text-textSecondary font-medium mb-1 block">Country</label>
                                <select x-model="computerForm.country"
                                        class="w-full rounded-2xl border border-borderColor/60 px-4 py-3 focus:ring-primary/40 focus:border-primary/50">
                                    <option value="">Select Country</option>
                                    <template x-for="option in (computerOptions.countries || [])" :key="option">
                                        <option :value="option" x-text="option"></option>
                                    </template>
                                </select>
                            </div>
                            <div>
                                <label class="text-sm text-textSecondary font-medium mb-1 block">Condition</label>
                                <select x-model="computerForm.condition"
                                        class="w-full rounded-2xl border border-borderColor/60 px-4 py-3 focus:ring-primary/40 focus:border-primary/50">
                                    <option value="">Select Condition</option>
                                    <template x-for="option in (computerOptions.conditions || [])" :key="option">
                                        <option :value="option" x-text="option"></option>
                                    </template>
                                </select>
                            </div>
                        </div>

                        <div class="grid md:grid-cols-2 gap-4 mt-4">
                            <div>
                                <label class="text-sm text-textSecondary font-medium mb-1 block">Activate</label>
                                <select x-model="computerForm.activate"
                                        class="w-full rounded-2xl border border-borderColor/60 px-4 py-3 focus:ring-primary/40 focus:border-primary/50">
                                    <option value="">Select Activate</option>
                                    <template x-for="option in (computerOptions.activates || [])" :key="option">
                                        <option :value="option" x-text="option"></option>
                                    </template>
                                </select>
                            </div>
                            <div>
                                <label class="text-sm text-textSecondary font-medium mb-1 block">Internal ID</label>
                                <input type="text" x-model="computerForm.internal_id"
                                       class="w-full rounded-2xl border border-borderColor/60 px-4 py-3 focus:ring-primary/40 focus:border-primary/50"
                                       placeholder="e.g. A13_1">
                            </div>
                        </div>

                        <div class="mt-4">
                            <label class="text-sm text-textSecondary font-medium mb-1 block">Description</label>
                            <input type="text" x-model="computerForm.description"
                                   class="w-full rounded-2xl border border-borderColor/60 px-4 py-3 focus:ring-primary/40 focus:border-primary/50"
                                   placeholder="Auto-generated on save if left blank">
                        </div>

                        <div class="rounded-2xl border border-red-200 bg-red-50 text-red-600 text-sm px-4 py-3 mt-4"
                             x-show="computerFormErrors.length">
                            <ul class="list-disc list-inside space-y-1">
                                <template x-for="(error, idx) in computerFormErrors" :key="idx">
                                    <li x-text="error"></li>
                                </template>
                            </ul>
                        </div>

                        <div class="flex justify-end mt-6">
                            <button type="button"
                                    class="px-6 py-3 rounded-2xl bg-primary text-white shadow shadow-primary/30"
                                    :disabled="computerFormLoading"
                                    @click="submitComputerForm">
                                <span x-show="!computerFormLoading">Create Product</span>
                                <span x-show="computerFormLoading">Saving...</span>
                            </button>
                        </div>
                    </div>

                    <div class="mt-6" x-show="computerModalTab === 'import'">
                        <div class="rounded-2xl border border-dashed border-borderColor/70 px-6 py-8 text-center">
                            <p class="text-sm text-textSecondary">Import CSV/XLSX with computer products.</p>
                            <input type="file" class="mt-4" x-ref="computerImportFile" accept=".csv,.xlsx">
                        </div>

                        <div class="rounded-2xl border border-red-200 bg-red-50 text-red-600 text-sm px-4 py-3 mt-4"
                             x-show="computerImportErrors.length">
                            <ul class="list-disc list-inside space-y-1">
                                <template x-for="(error, idx) in computerImportErrors" :key="idx">
                                    <li x-text="error"></li>
                                </template>
                            </ul>
                        </div>

                        <div class="flex justify-end mt-6">
                            <button type="button"
                                    class="px-6 py-3 rounded-2xl bg-primary text-white shadow shadow-primary/30"
                                    :disabled="computerImportLoading"
                                    @click="submitComputerImport">
                                <span x-show="!computerImportLoading">Import File</span>
                                <span x-show="computerImportLoading">Importing...</span>
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </form>

    <datalist id="phone-serials">
        @foreach($variations as $variation)
            @if($variation->serial_number)
                <option value="{{ $variation->serial_number }}">
                    {{ optional($variation->product)->name ?? 'Product #'.$variation->product_id }}
                </option>
            @endif
        @endforeach
    </datalist>

    @push('scripts')
        <script>
            document.addEventListener('alpine:init', () => {
                Alpine.data('purchaseForm', (payload = {}) => {
                    const defaults = payload.defaults || {};
                    const computerSearchUrl = '{{ route('computers.search') }}';
                    const computerStoreUrl = '{{ route('computers.store') }}';
                    const computerImportUrl = '{{ route('computers.import') }}';
                    const csrfToken = document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || '';
                    return {
                        step: payload.initialStep || 1,
                        purchaseType: payload.purchaseType || 'phone',
                        purchaseCode: payload.purchaseCode || '',
                        sInvoiceNumber: payload.sInvoiceNumber || '',
                        invoiceNumber: payload.invoiceNumber || '',
                        purchaseDate: payload.purchaseDate || '',
                        supplierId: payload.oldSupplier || '',
                        variationMeta: payload.variations || [],
                        computerOptions: payload.computerOptions || {},
                        phoneSerialMap: payload.phoneSerialMap || {},
                        taxAmount: Number(defaults.tax_amount ?? 0),
                        otherCharges: Number(defaults.other_charges ?? 0),
                        items: [],
                        showComputerModal: false,
                        computerModalTab: 'create',
                        computerForm: {},
                        computerFormErrors: [],
                        computerFormLoading: false,
                        computerImportLoading: false,
                        computerImportErrors: [],
                        activeComputerItemUid: null,
                        init() {
                            const previousItems = Array.isArray(payload.oldItems) ? payload.oldItems : [];
                            if (previousItems.length) {
                                this.items = previousItems.map(item => this.hydrateItem(item));
                            } else {
                                this.items = [this.newItem()];
                            }

                            this.items.forEach(item => this.syncSerials(item));

                            this.$watch('purchaseType', () => this.resetItemsForType());
                            this.resetComputerForm();
                        },
                        uid() {
                            return (Date.now().toString(36) + Math.random().toString(36).slice(2, 7));
                        },
                        hydrateItem(data) {
                            if (this.purchaseType === 'computer' || data.product_code) {
                                return this.hydrateComputerItem(data);
                            }
                            return this.hydratePhoneItem(data);
                        },
                        hydratePhoneItem(data) {
                            const variationId = data.variation_id ?? '';
                            const meta = this.lookupPhoneMeta(variationId);
                            const serials = Array.isArray(data.serial_numbers)
                                ? data.serial_numbers
                                : (data.serial_number ? [data.serial_number] : ['']);
                            return {
                                uid: this.uid(),
                                variation_id: variationId,
                                qty: Number(data.qty ?? 1),
                                cost_price: Number(data.cost_price ?? 0),
                                serial_numbers: serials.length ? serials : [''],
                                product_id: meta?.product_id || null,
                                details: meta || null,
                            };
                        },
                        hydrateComputerItem(data) {
                            const serials = Array.isArray(data.serial_numbers)
                                ? data.serial_numbers
                                : (data.serial_number ? [data.serial_number] : ['']);
                            return {
                                uid: this.uid(),
                                computer_product_id: data.computer_product_id || '',
                                product_code: data.product_code || '',
                                product_code_query: data.product_code || '',
                                code_name: data.code_name || '',
                                name: data.name || '',
                                display: data.display || '',
                                cpu: data.cpu || '',
                                capacity: data.capacity || '',
                                ram: data.ram || '',
                                storage: data.storage || '',
                                color_code: data.color_code || '',
                                color: data.color || '',
                                country_code: data.country_code || '',
                                country: data.country || '',
                                condition: data.condition || '',
                                activate: data.activate || '',
                                internal_id: data.internal_id || '',
                                description: data.description || '',
                                serial_numbers: serials.length ? serials : [''],
                                qty: Number(data.qty ?? 1) || 1,
                                cost_price: Number(data.cost_price ?? 0),
                                searchResults: [],
                                searching: false,
                                searchError: '',
                            };
                        },
                        newItem() {
                            return this.purchaseType === 'computer' ? this.newComputerItem() : this.newPhoneItem();
                        },
                        newPhoneItem() {
                            return {
                                uid: this.uid(),
                                variation_id: '',
                                qty: 1,
                                cost_price: 0,
                                serial_numbers: [''],
                                product_id: null,
                                details: null,
                            };
                        },
                        newComputerItem() {
                            return {
                                uid: this.uid(),
                                computer_product_id: '',
                                product_code: '',
                                product_code_query: '',
                                code_name: '',
                                name: '',
                                display: '',
                                cpu: '',
                                capacity: '',
                                ram: '',
                                storage: '',
                                color_code: '',
                                color: '',
                                country_code: '',
                                country: '',
                                condition: '',
                                activate: '',
                                internal_id: '',
                                description: '',
                                serial_numbers: [''],
                                qty: 1,
                                cost_price: 0,
                                searchResults: [],
                                searching: false,
                                searchError: '',
                            };
                        },
                        lookupPhoneMeta(variationId) {
                            if (!variationId) {
                                return null;
                            }
                            return this.variationMeta.find(v => String(v.id) === String(variationId)) || null;
                        },
                        syncSelection(item) {
                            if (this.purchaseType !== 'phone') return;
                            const meta = this.lookupPhoneMeta(item.variation_id);
                            item.product_id = meta?.product_id || null;
                            item.details = meta || null;
                        },
                        handleSerialChange(item, serialIndex) {
                            const serial = String(item.serial_numbers?.[serialIndex] || '').trim();
                            if (!serial) return;
                            if (this.isDuplicateSerial(item, serial, serialIndex)) {
                                alert('This serial number is already used in this purchase order.');
                                item.serial_numbers[serialIndex] = '';
                            }
                        },
                        isDuplicateSerial(item, serial, serialIndex) {
                            const normalized = String(serial).trim();
                            if (!normalized) return false;
                            const current = (item.serial_numbers || []).map(value => String(value || '').trim());
                            if (current.some((value, index) => index !== serialIndex && value === normalized)) {
                                return true;
                            }
                            return this.items.some((it) => {
                                if (it.uid === item.uid) return false;
                                return (it.serial_numbers || []).map(value => String(value || '').trim()).includes(normalized);
                            });
                        },
                        syncSerials(item) {
                            const qty = Math.max(1, Number(item.qty) || 1);
                            item.qty = qty;
                            if (!Array.isArray(item.serial_numbers)) {
                                item.serial_numbers = [''];
                            }
                            while (item.serial_numbers.length < qty) {
                                item.serial_numbers.push('');
                            }
                            while (item.serial_numbers.length > qty) {
                                item.serial_numbers.pop();
                            }
                        },
                        addSerial(item) {
                            item.qty = Number(item.qty || 0) + 1;
                            this.syncSerials(item);
                        },
                        removeSerial(item, serialIndex) {
                            if ((item.serial_numbers || []).length <= 1) {
                                return;
                            }
                            item.serial_numbers.splice(serialIndex, 1);
                            item.qty = Math.max(1, Number(item.qty || 1) - 1);
                            this.syncSerials(item);
                        },
                        serialsMismatch(item) {
                            const qty = Number(item.qty) || 0;
                            const filled = (item.serial_numbers || []).filter(value => String(value || '').trim() !== '').length;
                            return filled !== qty;
                        },
                        addItem() {
                            if (this.purchaseType === 'computer') {
                                this.items.push(this.newComputerItem());
                                return;
                            }

                            this.items.push(this.newPhoneItem());
                        },
                        removeItem(index) {
                            if (this.items.length === 1) return;
                            this.items.splice(index, 1);
                        },
                        resetItemsForType() {
                            if (this.purchaseType === 'computer') {
                                this.items = [this.newComputerItem()];
                            } else {
                                this.items = [this.newPhoneItem()];
                            }
                        },
                        normalizeComputerData(data) {
                            return {
                                id: data.id,
                                product_code: data.product_code,
                                code_name: data.code_name,
                                name: data.name,
                                display: data.display,
                                cpu: data.cpu,
                                capacity: data.capacity,
                                ram: data.ram,
                                storage: data.storage,
                                color_code: data.color_code,
                                color: data.color,
                                country_code: data.country_code,
                                country: data.country,
                                condition: data.condition,
                                activate: data.activate,
                                internal_id: data.internal_id,
                                description: data.description,
                                serial_number: data.serial_number,
                                cost: data.cost,
                            };
                        },
                        searchComputers(item) {
                            const query = String(item.product_code_query || '').trim();
                            if (query !== item.product_code) {
                                this.clearComputerSelection(item);
                            }
                            item.searchError = '';
                            if (!query) {
                                item.searchResults = [];
                                item.searching = false;
                                return;
                            }
                            item.searching = true;
                            fetch(`${computerSearchUrl}?query=${encodeURIComponent(query)}`, {
                                headers: { 'X-Requested-With': 'XMLHttpRequest' }
                            })
                                .then(response => response.ok ? response.json() : Promise.reject(response))
                                .then(data => {
                                    item.searchResults = Array.isArray(data.data) ? data.data : [];
                                })
                                .catch(() => {
                                    item.searchError = 'Unable to load inventory. Try again.';
                                    item.searchResults = [];
                                })
                                .finally(() => {
                                    item.searching = false;
                                });
                        },
                        selectComputer(item, computer) {
                            const meta = this.normalizeComputerData(computer);
                            item.computer_product_id = meta.id;
                            item.product_code = meta.product_code || '';
                            item.product_code_query = meta.product_code || '';
                            item.code_name = meta.code_name || '';
                            item.name = meta.name || '';
                            item.display = meta.display || '';
                            item.cpu = meta.cpu || '';
                            item.capacity = meta.capacity || '';
                            item.ram = meta.ram || '';
                            item.storage = meta.storage || '';
                            item.color_code = meta.color_code || '';
                            item.color = meta.color || '';
                            item.country_code = meta.country_code || '';
                            item.country = meta.country || '';
                            item.condition = meta.condition || '';
                            item.activate = meta.activate || '';
                            item.internal_id = meta.internal_id || '';
                            item.description = meta.description || '';
                            item.serial_numbers = [''];
                            this.syncSerials(item);
                            item.searchResults = [];
                            if (meta.cost !== undefined) {
                                item.cost_price = Number(meta.cost) || item.cost_price;
                            }
                        },
                        clearComputerSelection(item) {
                            item.computer_product_id = '';
                            item.product_code = '';
                            item.code_name = '';
                            item.name = '';
                            item.display = '';
                            item.cpu = '';
                            item.capacity = '';
                            item.ram = '';
                            item.storage = '';
                            item.color_code = '';
                            item.color = '';
                            item.country_code = '';
                            item.country = '';
                            item.condition = '';
                            item.activate = '';
                            item.internal_id = '';
                            item.description = '';
                            item.serial_numbers = [''];
                            item.searchResults = [];
                        },
                        openComputerModal(item) {
                            this.activeComputerItemUid = item.uid;
                            this.computerModalTab = 'create';
                            this.showComputerModal = true;
                            this.resetComputerForm();
                        },
                        resetComputerForm() {
                            this.computerForm = {
                                description: '',
                                code_name: '',
                                name: '',
                                display: '',
                                cpu: '',
                                capacity: '',
                                ram: '',
                                storage: '',
                                color_code: '',
                                color: '',
                                country_code: '',
                                country: '',
                                condition: '',
                                activate: '',
                                internal_id: '',
                            };
                            this.computerFormErrors = [];
                            this.computerFormLoading = false;
                            this.computerImportLoading = false;
                            this.computerImportErrors = [];
                        },
                        submitComputerForm() {
                            this.computerFormErrors = [];
                            this.computerFormLoading = true;
                            fetch(computerStoreUrl, {
                                method: 'POST',
                                headers: {
                                    'Content-Type': 'application/json',
                                    'Accept': 'application/json',
                                    'X-CSRF-TOKEN': csrfToken
                                },
                                body: JSON.stringify(this.computerForm)
                            })
                                .then(async (response) => {
                                    const data = await response.json().catch(() => ({}));
                                    if (!response.ok) {
                                        const errors = data.errors ? Object.values(data.errors).flat() : [data.message || 'Unable to create computer product.'];
                                        throw errors;
                                    }
                                    return data;
                                })
                                .then((data) => {
                                    const computer = data.data || data;
                                    const item = this.items.find(i => i.uid === this.activeComputerItemUid);
                                    if (item) {
                                        this.selectComputer(item, computer);
                                    }
                                    this.showComputerModal = false;
                                })
                                .catch((errors) => {
                                    this.computerFormErrors = Array.isArray(errors) ? errors : ['Unable to create computer product.'];
                                })
                                .finally(() => {
                                    this.computerFormLoading = false;
                                });
                        },
                        submitComputerImport() {
                            const file = this.$refs?.computerImportFile?.files?.[0];
                            if (!file) {
                                this.computerImportErrors = ['Please select a file to import.'];
                                return;
                            }
                            const formData = new FormData();
                            formData.append('import_file', file);
                            this.computerImportErrors = [];
                            this.computerImportLoading = true;
                            fetch(computerImportUrl, {
                                method: 'POST',
                                headers: {
                                    'Accept': 'application/json',
                                    'X-CSRF-TOKEN': csrfToken
                                },
                                body: formData
                            })
                                .then(async (response) => {
                                    const data = await response.json().catch(() => ({}));
                                    if (!response.ok) {
                                        const errors = data.errors ? Object.values(data.errors).flat() : [data.message || 'Unable to import computer products.'];
                                        throw errors;
                                    }
                                    return data;
                                })
                                .then(() => {
                                    this.showComputerModal = false;
                                    if (this.$refs?.computerImportFile) {
                                        this.$refs.computerImportFile.value = '';
                                    }
                                })
                                .catch((errors) => {
                                    this.computerImportErrors = Array.isArray(errors) ? errors : ['Unable to import computer products.'];
                                })
                                .finally(() => {
                                    this.computerImportLoading = false;
                                });
                        },
                        formatCurrency(value) {
                            const number = Number(value) || 0;
                            return new Intl.NumberFormat('en-US', {
                                style: 'currency',
                                currency: 'USD'
                            }).format(number);
                        },
                        formatVariant(details) {
                            if (!details) return '-';
                            return [details.color, details.storage].filter(Boolean).join(' / ') || '-';
                        },
                        buildComputerSummary(item) {
                            if (!item) return '';
                            const parts = [
                                [item.display, item.cpu].filter(Boolean).join(' / '),
                                [item.capacity, item.ram, item.storage].filter(Boolean).join(' / '),
                                [item.color_code, item.color].filter(Boolean).join(' '),
                                [item.country_code, item.country].filter(Boolean).join(' '),
                                item.condition,
                                item.activate,
                            ].filter(Boolean);
                            return parts.join(' | ');
                        },
                        buildPhoneSummary(item) {
                            const details = item?.details;
                            if (!details) return '';
                            const parts = [
                                details.sku ? `SKU ${details.sku}` : null,
                                this.formatVariant(details),
                            ].filter(Boolean);
                            return parts.join(' | ');
                        },
                        itemSummary(item) {
                            return this.purchaseType === 'computer'
                                ? this.buildComputerSummary(item)
                                : this.buildPhoneSummary(item);
                        },
                        summaryTitle(item) {
                            return this.purchaseType === 'computer'
                                ? (item.name || item.product_code || 'Product details')
                                : (item.details?.product_name || 'Product details');
                        },
                        nextStep() {
                            if (this.step < 3 && this.canProceed) {
                                this.step += 1;
                            }
                        },
                        prevStep() {
                            if (this.step > 1) {
                                this.step -= 1;
                            }
                        },
                        canProceedStep1() {
                            return Boolean(this.invoiceNumber && this.supplierId && this.purchaseDate && this.purchaseType);
                        },
                        canProceedStep2() {
                            return this.items.length > 0 && this.items.every(item => {
                                const hasQty = Number(item.qty) > 0;
                                const hasCost = Number(item.cost_price) > 0;
                                const serialOk = !this.serialsMismatch(item);

                                if (this.purchaseType === 'phone') {
                                    return Boolean(item.variation_id) && hasQty && hasCost && serialOk;
                                }

                                return Boolean(item.computer_product_id) && hasQty && hasCost && serialOk;
                            });
                        },
                        get canProceed() {
                            if (this.step === 1) {
                                return this.canProceedStep1();
                            }
                            if (this.step === 2) {
                                return this.canProceedStep2();
                            }
                            return true;
                        },
                        get totalAmount() {
                            return this.items.reduce((total, item) => {
                                return total + ((Number(item.qty) || 0) * (Number(item.cost_price) || 0));
                            }, 0);
                        },
                        get totalDue() {
                            return this.totalAmount + (Number(this.taxAmount) || 0) + (Number(this.otherCharges) || 0);
                        }
                    };
                });
            });
        </script>
    @endpush


</x-app-layout>
