<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\Branch;
use Illuminate\Http\RedirectResponse;
use Illuminate\Support\Facades\Hash;
use Illuminate\Contracts\View\View;
use Illuminate\Support\Carbon;
use Illuminate\Http\Request;
use Spatie\Permission\Models\Permission;
use Spatie\Permission\Models\Role;

class SettingsController extends Controller
{
    public function index(): View
    {
        $roles = Role::withCount('users')->get();
        $permissions = Permission::withCount('roles')->get();
        $users = User::with('roles')->get()->map(function ($user) {
            $lastActive = $user->updated_at ?? $user->created_at ?? now();
            $user->is_online = $lastActive->gt(Carbon::now()->subMinutes(5));
            return $user;
        });

        return view('settings.index', compact('roles', 'permissions', 'users'));
    }

    public function users(): View
    {
        $users = User::with('roles', 'branch')->get();
        $roles = Role::all();
        $branches = Branch::all();

        return view('settings.users', compact('users', 'roles', 'branches'));
    }

    public function storeUser(Request $request): RedirectResponse
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email',
            'password' => 'required|string|min:8',
            'branch_id' => 'nullable|exists:branches,id',
            'roles' => 'array',
            'roles.*' => 'string|exists:roles,name',
        ]);

        $user = User::create([
            'name' => $validated['name'],
            'email' => $validated['email'],
            'password' => Hash::make($validated['password']),
            'branch_id' => $validated['branch_id'],
        ]);

        $user->syncRoles($validated['roles'] ?? []);

        return back()->with('status', 'User created successfully.');
    }

    public function updateUserRoles(Request $request, User $user): RedirectResponse
    {
        $validated = $request->validate([
            'roles' => 'array',
            'roles.*' => 'string|exists:roles,name',
        ]);

        $user->syncRoles($validated['roles'] ?? []);

        return back()->with('status', 'User roles updated.');
    }
}
