<?php

namespace App\Http\Controllers;

use App\Models\Supplier;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use ZipArchive;

class SupplierController extends Controller
{
    public function index()
    {
        $suppliers = Supplier::paginate(20);
        return view('suppliers.index', compact('suppliers'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'phone' => ['nullable', 'string', 'max:50'],
            'email' => ['nullable', 'email', 'max:255'],
            'address' => ['nullable', 'string', 'max:255'],
        ]);

        Supplier::create($validated);

        return back()->with('status', 'Supplier added successfully.');
    }

    public function import(Request $request)
    {
        $request->validate([
            'file' => ['required', 'file', 'mimes:csv,txt,xlsx'],
        ]);

        $file = $request->file('file');
        $extension = strtolower($file->getClientOriginalExtension());

        try {
            if ($extension === 'csv' || $extension === 'txt') {
                $rows = $this->parseCsv($file->getRealPath());
            } else {
                $rows = $this->parseXlsx($file->getRealPath());
            }
        } catch (\Throwable $e) {
            Log::error('Supplier import failed', ['error' => $e->getMessage()]);
            return back()->withErrors(['file' => 'Unable to read the uploaded file. Please check the format and try again.']);
        }

        if (empty($rows)) {
            return back()->withErrors(['file' => 'No data detected in the uploaded file.']);
        }

        $headers = array_map(function ($header) {
            return Str::slug((string) $header, '_');
        }, array_shift($rows));

        $created = 0;
        foreach ($rows as $row) {
            if ($this->createSupplierFromRow($headers, $row)) {
                $created++;
            }
        }

        if ($created === 0) {
            return back()->withErrors(['file' => 'No suppliers were imported. Ensure your file has name, phone, email, and address columns.']);
        }

        return back()->with('status', "{$created} supplier(s) imported successfully.");
    }

    protected function createSupplierFromRow(array $headers, array $row): bool
    {
        $data = [];
        foreach ($headers as $index => $header) {
            $data[$header] = isset($row[$index]) ? trim((string) $row[$index]) : null;
        }

        $name = $data['name'] ?? $data['supplier_name'] ?? null;
        if (! $name) {
            return false;
        }

        Supplier::create([
            'name' => $name,
            'phone' => $data['phone'] ?? null,
            'email' => $data['email'] ?? null,
            'address' => $data['address'] ?? ($data['location'] ?? null),
        ]);

        return true;
    }

    protected function parseCsv(string $path): array
    {
        $rows = [];
        if (($handle = fopen($path, 'r')) !== false) {
            while (($data = fgetcsv($handle)) !== false) {
                if (! $this->rowIsEmpty($data)) {
                    $rows[] = $data;
                }
            }
            fclose($handle);
        }
        return $rows;
    }

    protected function parseXlsx(string $path): array
    {
        $zip = new ZipArchive();
        if ($zip->open($path) !== true) {
            throw new \RuntimeException('Unable to open XLSX file.');
        }

        $sheet = $zip->getFromName('xl/worksheets/sheet1.xml');
        if ($sheet === false) {
            throw new \RuntimeException('Worksheet not found.');
        }
        $sheetXml = simplexml_load_string($sheet, null, LIBXML_NOCDATA);
        $namespace = 'http://schemas.openxmlformats.org/spreadsheetml/2006/main';
        $sheetXml->registerXPathNamespace('ns', $namespace);

        $sharedStrings = [];
        if ($shared = $zip->getFromName('xl/sharedStrings.xml')) {
            $sharedXml = simplexml_load_string($shared, null, LIBXML_NOCDATA);
            $sharedXml->registerXPathNamespace('ns', $namespace);
            foreach ($sharedXml->xpath('//ns:si') as $si) {
                $textNodes = $si->xpath('.//ns:t') ?: [];
                $string = '';
                foreach ($textNodes as $t) {
                    $string .= (string) $t;
                }
                $sharedStrings[] = $string;
            }
        }

        $rows = [];
        $rowNodes = $sheetXml->xpath('//ns:sheetData/ns:row') ?: [];
        foreach ($rowNodes as $row) {
            $rowData = [];
            foreach ($row->children($namespace) as $c) {
                if ($c->getName() !== 'c') {
                    continue;
                }
                $value = '';
                $type = (string) $c['t'];
                $cellRef = (string) $c['r'];
                $columnIndex = $this->columnLettersToIndex(preg_replace('/\d+/', '', $cellRef));

                if ($c->v !== null) {
                    $index = (int) $c->v;
                    if ($type === 's') {
                        $value = $sharedStrings[$index] ?? '';
                    } else {
                        $value = (string) $c->v;
                    }
                }

                $rowData[$columnIndex] = $value;
            }

            if (! empty($rowData)) {
                ksort($rowData);
                $maxIndex = array_key_last($rowData);
                $normalized = [];
                for ($i = 0; $i <= $maxIndex; $i++) {
                    $normalized[] = $rowData[$i] ?? '';
                }

                if (! $this->rowIsEmpty($normalized)) {
                    $rows[] = $normalized;
                }
            }
        }

        return $rows;
    }

    protected function rowIsEmpty(array $row): bool
    {
        foreach ($row as $value) {
            if (trim((string) $value) !== '') {
                return false;
            }
        }
        return true;
    }

    protected function columnLettersToIndex(string $letters): int
    {
        $letters = strtoupper($letters);
        $index = 0;
        $length = strlen($letters);
        for ($i = 0; $i < $length; $i++) {
            $index = $index * 26 + (ord($letters[$i]) - 64);
        }

        return max(0, $index - 1);
    }
}
