<?php

namespace App\Http\Controllers;

use App\Helpers\NumberHelper;
use App\Models\Repair;
use Illuminate\Http\Request;

class RepairController extends Controller
{
    protected array $statuses = ['pending', 'in_progress', 'completed', 'delivered'];

    protected array $issueOptions = [
        'Power & Charging Issues' => [
            'Device not powering on',
            'Device restarts randomly',
            'Battery draining fast',
            'Battery not charging',
            'Slow charging',
            'Charger not detected',
            'Overheating',
            'Charging port loose or damaged',
            'Wireless charging not working',
        ],
        'Display & Touch Issues' => [
            'Screen cracked',
            'Touch screen not responding',
            'Ghost touch / auto-touch',
            'Screen flickering',
            'Black screen',
            'Lines on display',
            'Yellow/green/white spots',
            'Backlight not working',
            'LCD bleeding',
        ],
        'Audio Issues' => [
            'No sound',
            'Ear speaker not working',
            'Loudspeaker not working',
            'Microphone not working',
            'Distorted audio',
            'Low volume',
            'Headphones not detected',
        ],
        'Camera Issues' => [
            'Back camera not working',
            'Front camera not working',
            'Blurry images',
            'Camera app crashing',
            'Flashlight not working',
            'Camera shaking / OIS failure',
        ],
        'Network & Connectivity Issues' => [
            'No SIM detection',
            'Weak signal',
            'No service',
            'WiFi not working',
            'Bluetooth not working',
            'Hotspot not working',
            'GPS not accurate',
        ],
        'Software Issues' => [
            'Boot loop issue',
            'Apple logo stuck',
            'Android logo stuck',
            'App crashing',
            'System slow/lagging',
            'iCloud lock issue',
            'Google FRP lock',
            'Data recovery needed',
            'OS update failure',
        ],
        'Button & Control Issues' => [
            'Power button not working',
            'Volume button not working',
            'Home button not working',
            'Mute switch not working',
            'Fingerprint not working (Touch ID)',
            'Face ID not working',
        ],
        'Battery & Heating Issues' => [
            'Battery swelling',
            'Device overheating',
            'Battery percentage jumping',
            'Device auto-shutdown',
        ],
        'Water/Liquid Damage' => [
            'Water-damaged device',
            'No display after water damage',
            'No power after liquid exposure',
            'Corrosion on board',
        ],
        'Other Hardware Issues' => [
            'Vibration motor not working',
            'Proximity sensor not working',
            'Light sensor not working',
            'Back glass cracked',
            'Frame bent',
            'SIM tray jammed',
            'Speaker mesh dirty',
        ],
        'Logic Board / IC Issues' => [
            'No backlight (Backlight IC)',
            'No touch (Touch IC)',
            'No charging (Charging IC)',
            'Audio IC issue',
            'Baseband IC issue',
            'Power IC issue',
            'NAND/Storage chip issue',
        ],
    ];

    public function index()
    {
        $repairs = Repair::latest()->paginate(20);
        return view('repairs.index', compact('repairs'));
    }

    public function create()
    {
        $statuses = $this->statuses;
        $issueOptions = $this->issueOptions;
        return view('repairs.create', compact('statuses', 'issueOptions'));
    }

    public function store(Request $request)
    {
        $statuses = $this->statuses;

        $validated = $request->validate([
            'device_model' => 'required|string|max:255',
            'customer_id' => 'nullable|integer',
            'issues' => 'nullable|array',
            'issues.*' => 'string',
            'issue_custom' => 'nullable|string',
            'status' => 'nullable|in:'.implode(',', $statuses),
        ]);

        $issueText = $this->formatIssues($validated['issues'] ?? [], $validated['issue_custom'] ?? null);
        if (!$issueText) {
            return back()->withErrors(['issues' => 'Please select or enter at least one issue.'])->withInput();
        }

        $autoCustomerId = (int) now()->format('ymdHis');
        $customerId = $request->filled('customer_id') ? (int) $request->customer_id : $autoCustomerId;

        $repair = Repair::create([
            'repair_code' => NumberHelper::generate('RP', new Repair),
            'customer_id' => $customerId,
            'device_model' => $validated['device_model'],
            'issue' => $issueText,
            'status' => $validated['status'] ?? 'pending',
        ]);

        return redirect()->route('repairs.show', $repair->id);
    }

    public function show(Repair $repair)
    {
        $repair->load('items');
        return view('repairs.show', compact('repair'));
    }

    public function edit(Repair $repair)
    {
        $statuses = $this->statuses;
        $issueOptions = $this->issueOptions;
        $existingIssues = collect($this->splitIssues($repair->issue));
        $presetIssues = collect($this->issueOptions)->flatMap(fn ($issues) => $issues);
        $selectedIssues = $existingIssues->filter(fn ($issue) => $presetIssues->contains($issue))->values()->all();
        $customIssuePrefill = $existingIssues->reject(fn ($issue) => $presetIssues->contains($issue))->implode("\n");

        return view('repairs.edit', compact('repair', 'statuses', 'issueOptions', 'selectedIssues', 'customIssuePrefill'));
    }

    public function update(Request $request, Repair $repair)
    {
        $statuses = $this->statuses;

        $validated = $request->validate([
            'device_model' => 'required|string|max:255',
            'customer_id' => 'nullable|integer',
            'issues' => 'nullable|array',
            'issues.*' => 'string',
            'issue_custom' => 'nullable|string',
            'status' => 'required|in:'.implode(',', $statuses),
        ]);

        $issueText = $this->formatIssues($validated['issues'] ?? [], $validated['issue_custom'] ?? null);
        if (!$issueText) {
            return back()->withErrors(['issues' => 'Please select or enter at least one issue.'])->withInput();
        }

        $repair->update([
            'device_model' => $validated['device_model'],
            'customer_id' => $validated['customer_id'] ?? $repair->customer_id,
            'issue' => $issueText,
            'status' => $validated['status'],
        ]);

        return redirect()->route('repairs.show', $repair->id)->with('success', 'Repair updated.');
    }

    public function destroy(Repair $repair)
    {
        $repair->delete();

        return redirect()->route('repairs.index')->with('success', 'Repair deleted.');
    }

    public function updateStatus(Request $request, Repair $repair)
    {
        $request->validate(['status' => 'required|in:'.implode(',', $this->statuses)]);
        $repair->update(['status' => $request->status]);

        return back();
    }

    protected function formatIssues(array $issues, ?string $custom): ?string
    {
        $customLines = collect(preg_split('/\r\n|\r|\n/', (string) $custom));

        $combined = collect($issues)
            ->merge($customLines)
            ->map(fn ($issue) => trim($issue))
            ->filter();

        if ($combined->isEmpty()) {
            return null;
        }

        return $combined->map(fn ($issue) => '• '.$issue)->implode("\n");
    }

    protected function splitIssues(?string $issueText): array
    {
        return collect(preg_split('/\r\n|\r|\n/', $issueText ?? ''))
            ->map(fn ($line) => ltrim($line, "• \t"))
            ->filter()
            ->values()
            ->all();
    }
}
