<?php

namespace App\Http\Controllers;

use App\Models\Product;
use App\Models\ProductVariation;
use App\Models\StockMovement;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Illuminate\Validation\ValidationException;
use PhpOffice\PhpSpreadsheet\IOFactory;

class ProductController extends Controller
{
    public function index()
    {
        $products = Product::with('variations')->paginate(20);
        return view('products.index', compact('products'));
    }

    public function create()
    {
        $nextSku = ProductVariation::nextSku();
        $nextCategoryCode = Product::nextCategoryCode();

        return view('products.create', compact('nextSku', 'nextCategoryCode'));
    }

    public function store(Request $request)
    {
        $colorOptions = config('products.colors', ['Black', 'Silver', 'White']);
        $storageOptions = config('products.storages', ['32GB', '64GB', '128GB', '256GB', '512GB', '1TB']);
        $conditionOptions = array_keys(config('products.conditions', ['SC' => 'SC', 'New' => 'New']));
        $networkOptions = array_keys(config('products.network_statuses', ['AT' => 'AT', 'NT' => 'NT']));

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'selling_price' => 'required|numeric|min:0',
            'cost_price' => 'required|numeric|min:0',
            'is_active' => 'nullable|boolean',
            'color' => ['required', 'string', 'max:255', Rule::in($colorOptions)],
            'storage' => ['nullable', 'string', 'max:255', Rule::in($storageOptions)],
            'barcode' => 'nullable|string|max:255',
            'initial_stock' => 'nullable|integer|min:0',
            'imei' => 'nullable|string|max:255|unique:product_variations,imei',
            'serial_number' => 'nullable|string|max:255|unique:product_variations,serial_number',
            'condition' => ['required', Rule::in($conditionOptions)],
            'network_status' => ['nullable', Rule::in($networkOptions)],
        ]);

        $categoryCode = Product::nextCategoryCode();
        $sku = ProductVariation::nextSku();

        $product = Product::create([
            'name' => $validated['name'],
            'category_code' => $categoryCode,
            'selling_price' => $validated['selling_price'],
            'cost_price' => $validated['cost_price'],
            'is_active' => $request->boolean('is_active', true),
        ]);

        $variation = ProductVariation::create([
            'product_id' => $product->id,
            'sku' => $sku,
            'color' => $validated['color'],
            'storage' => $validated['storage'] ?? null,
            'barcode' => $validated['barcode'] ?? null,
            'imei' => $validated['imei'] ?? null,
            'serial_number' => $validated['serial_number'] ?? null,
            'condition' => $validated['condition'],
            'network_status' => $validated['network_status'] ?? null,
        ]);

        $initialStock = $validated['initial_stock'] ?? null;
        $branchId = auth()->user()->branch_id ?? null;

        if ($initialStock && $branchId) {
            StockMovement::create([
                'branch_id' => $branchId,
                'variation_id' => $variation->id,
                'type' => 'adjustment',
                'qty' => $initialStock,
                'reference' => 'INITIAL_STOCK',
            ]);
        }

        return redirect()->route('products.index')->with('success', 'Product created');
    }

    public function import(Request $request)
    {
        $request->validate([
            'import_file' => 'required|file|mimetypes:text/plain,text/csv,application/vnd.ms-excel,application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
        ]);

        $rows = $this->parseImportFile($request->file('import_file')->getRealPath(), $request->file('import_file')->getClientOriginalExtension());

        $header = array_shift($rows);
        if (! $header) {
            return back()->withErrors(['import_file' => 'The uploaded file is empty.']);
        }

        $header = array_map(fn ($value) => Str::of($value)->trim()->snake()->value(), $header);

        $branchId = auth()->user()->branch_id;
        $rowsImported = 0;

        DB::transaction(function () use ($rows, $header, $branchId, &$rowsImported) {
            foreach ($rows as $row) {
                if (count(array_filter($row)) === 0) {
                    continue;
                }

                $data = array_combine($header, $row);
                if (! $data || empty($data['name'])) {
                    continue;
                }

                $sellingPrice = (float) ($data['selling_price'] ?? 0);
                $costPrice = (float) ($data['cost_price'] ?? 0);
                $isActive = isset($data['is_active']) ? filter_var($data['is_active'], FILTER_VALIDATE_BOOLEAN, FILTER_NULL_ON_FAILURE) : true;
                $initialStock = (int) ($data['initial_stock'] ?? 0);

                $product = Product::create([
                    'name' => $data['name'],
                    'category_code' => Product::nextCategoryCode(),
                    'selling_price' => $sellingPrice,
                    'cost_price' => $costPrice,
                    'is_active' => $isActive ?? true,
                ]);

                $variation = ProductVariation::create([
                    'product_id' => $product->id,
                    'sku' => ProductVariation::nextSku(),
                    'color' => $data['color'] ?? null,
                    'storage' => $data['storage'] ?? null,
                    'barcode' => $data['barcode'] ?? null,
                    'imei' => $data['imei'] ?? null,
                    'serial_number' => $data['serial_number'] ?? null,
                    'condition' => $data['condition'] ?? 'New',
                    'network_status' => $data['network_status'] ?? null,
                ]);

                if ($initialStock > 0 && $branchId) {
                    StockMovement::create([
                        'branch_id' => $branchId,
                        'variation_id' => $variation->id,
                        'type' => 'adjustment',
                        'qty' => $initialStock,
                        'reference' => 'IMPORT',
                    ]);
                }

                $rowsImported++;
            }
        });

        return back()->with('success', "Imported {$rowsImported} products.");
    }

    protected function parseImportFile(string $path, ?string $extension): array
    {
        $extension = strtolower($extension ?? '');

        if ($extension === 'csv' || str_contains(mime_content_type($path), 'csv')) {
            return $this->parseCsv($path);
        }

        if (in_array($extension, ['xlsx', 'xls'])) {
            if (! class_exists(IOFactory::class)) {
                throw ValidationException::withMessages([
                    'import_file' => 'Excel imports require phpoffice/phpspreadsheet. Please install it or upload a CSV file.',
                ]);
            }

            $reader = IOFactory::createReader('Xlsx');
            $spreadsheet = $reader->load($path);
            $worksheet = $spreadsheet->getActiveSheet();
            return $worksheet->toArray();
        }

        throw ValidationException::withMessages([
            'import_file' => 'Unsupported file type. Please upload a CSV or Excel (.xlsx) file.',
        ]);
    }

    protected function parseCsv(string $path): array
    {
        $rows = [];
        if (($handle = fopen($path, 'r')) !== false) {
            while (($row = fgetcsv($handle, 0, ',')) !== false) {
                $rows[] = $row;
            }
            fclose($handle);
        }

        return $rows;
    }

    public function show($id)
    {
        $product = Product::with('variations')->findOrFail($id);
        $nextSku = ProductVariation::nextSku();

        return view('products.show', compact('product', 'nextSku'));
    }
}
