<?php

namespace App\Http\Controllers;

use App\Models\ComputerProduct;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Illuminate\Validation\ValidationException;
use PhpOffice\PhpSpreadsheet\IOFactory;

class ComputerProductController extends Controller
{
    public function index()
    {
        $computers = ComputerProduct::latest()->get();
        return view('computers.index', compact('computers'));
    }

    public function create()
    {
        $nextCode = $this->nextProductCode();
        return view('computers.create', compact('nextCode'));
    }

    public function store(Request $request)
    {
        $options = config('computers');

        $validated = $request->validate([
            'product_code' => ['required', 'string', 'max:255', Rule::unique('computer_products', 'product_code')],
            'name' => ['required', 'string', 'max:255'],
            'display' => ['required', Rule::in($options['displays'] ?? [])],
            'cpu' => ['required', Rule::in($options['cpus'] ?? [])],
            'capacity' => ['required', Rule::in($options['capacities'] ?? [])],
            'ram' => ['required', Rule::in($options['rams'] ?? [])],
            'storage' => ['required', Rule::in($options['storages'] ?? [])],
            'color' => ['required', Rule::in($options['colors'] ?? [])],
            'country_code' => ['required', Rule::in($options['countries'] ?? [])],
            'condition' => ['required', Rule::in($options['conditions'] ?? [])],
            'status' => ['required', Rule::in($options['statuses'] ?? [])],
            'serial_number' => ['required', 'string', 'max:10'],
            'cost' => ['required', 'numeric', 'min:0'],
            'tax' => ['nullable', 'numeric', 'min:0'],
            'stock_qty' => ['required', 'integer', 'min:0'],
        ]);

        $validated['tax'] = $validated['tax'] ?? 0;

        ComputerProduct::create($validated);

        return redirect()->route('computers.index')->with('success', 'Computer product created.');
    }

    protected function nextProductCode(): string
    {
        $latest = ComputerProduct::orderByDesc('id')->first();
        $nextNumber = $latest ? ((int) preg_replace('/[^0-9]/', '', $latest->product_code)) + 1 : 1;

        return 'MB-' . str_pad((string) $nextNumber, 3, '0', STR_PAD_LEFT);
    }

    public function show(ComputerProduct $computer)
    {
        return view('computers.show', compact('computer'));
    }

    public function edit(ComputerProduct $computer)
    {
        $nextCode = $computer->product_code;
        return view('computers.edit', compact('computer', 'nextCode'));
    }

    public function update(Request $request, ComputerProduct $computer)
    {
        $options = config('computers');

        $validated = $request->validate([
            'product_code' => ['required', 'string', 'max:255', Rule::unique('computer_products', 'product_code')->ignore($computer->id)],
            'name' => ['required', 'string', 'max:255'],
            'display' => ['required', Rule::in($options['displays'] ?? [])],
            'cpu' => ['required', Rule::in($options['cpus'] ?? [])],
            'capacity' => ['required', Rule::in($options['capacities'] ?? [])],
            'ram' => ['required', Rule::in($options['rams'] ?? [])],
            'storage' => ['required', Rule::in($options['storages'] ?? [])],
            'color' => ['required', Rule::in($options['colors'] ?? [])],
            'country_code' => ['required', Rule::in($options['countries'] ?? [])],
            'condition' => ['required', Rule::in($options['conditions'] ?? [])],
            'status' => ['required', Rule::in($options['statuses'] ?? [])],
            'serial_number' => ['required', 'string', 'max:10', Rule::unique('computer_products', 'serial_number')->ignore($computer->id)],
            'cost' => ['required', 'numeric', 'min:0'],
            'tax' => ['nullable', 'numeric', 'min:0'],
            'stock_qty' => ['required', 'integer', 'min:0'],
        ]);

        $validated['tax'] = $validated['tax'] ?? 0;

        $computer->update($validated);

        return redirect()->route('computers.show', $computer)->with('success', 'Computer product updated.');
    }

    public function destroy(ComputerProduct $computer)
    {
        $computer->delete();
        return redirect()->route('computers.index')->with('success', 'Computer product deleted.');
    }

    public function import(Request $request)
    {
        $request->validate([
            'import_file' => 'required|file|mimetypes:text/plain,text/csv,application/vnd.ms-excel,application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
        ]);

        $file = $request->file('import_file');

        $rows = $this->parseImportFile($file->getRealPath(), $file->getClientOriginalExtension());

        $header = array_shift($rows);
        if (! $header) {
            return back()->withErrors(['import_file' => 'The uploaded file is empty.']);
        }

        $header = array_map(fn ($value) => Str::of($value)->trim()->snake()->value(), $header);

        $rowsImported = 0;

        DB::transaction(function () use (&$rowsImported, $rows, $header) {
            foreach ($rows as $row) {
                if (count(array_filter($row)) === 0) {
                    continue;
                }

                $data = array_combine($header, $row);
                if (! $data) {
                    continue;
                }

                $payload = [
                    'product_code' => $data['product_code'] ?? $this->nextProductCode(),
                    'name' => $data['name'] ?? 'Unnamed',
                    'display' => $data['display'] ?? '',
                    'cpu' => $data['cpu'] ?? '',
                    'capacity' => $data['capacity'] ?? '',
                    'ram' => $data['ram'] ?? '',
                    'storage' => $data['storage'] ?? '',
                    'color' => $data['color'] ?? '',
                    'country_code' => $data['country_code'] ?? '',
                    'condition' => $data['condition'] ?? 'New',
                    'status' => $data['status'] ?? 'Active',
                    'serial_number' => $data['serial_number'] ?? '',
                    'cost' => (float) ($data['cost'] ?? 0),
                    'tax' => (float) ($data['tax'] ?? 0),
                    'stock_qty' => (int) ($data['stock_qty'] ?? 0),
                ];

                ComputerProduct::updateOrCreate(
                    ['product_code' => $payload['product_code']],
                    $payload
                );

                $rowsImported++;
            }
        });

        return back()->with('success', "Imported {$rowsImported} computer products.");
    }

    protected function parseImportFile(string $path, ?string $extension): array
    {
        $extension = strtolower($extension ?? '');

        if ($extension === 'csv' || str_contains(mime_content_type($path), 'csv')) {
            return $this->parseCsv($path);
        }

        if (in_array($extension, ['xlsx', 'xls'])) {
            if (! class_exists(IOFactory::class)) {
                throw ValidationException::withMessages([
                    'import_file' => 'Excel imports require phpoffice/phpspreadsheet. Please install it or upload a CSV file.',
                ]);
            }

            $reader = IOFactory::createReader('Xlsx');
            $spreadsheet = $reader->load($path);
            $worksheet = $spreadsheet->getActiveSheet();
            return $worksheet->toArray();
        }

        throw ValidationException::withMessages([
            'import_file' => 'Unsupported file type. Please upload a CSV or Excel (.xlsx) file.',
        ]);
    }

    protected function parseCsv(string $path): array
    {
        $rows = [];
        if (($handle = fopen($path, 'r')) !== false) {
            while (($row = fgetcsv($handle, 0, ',')) !== false) {
                $rows[] = $row;
            }
            fclose($handle);
        }

        return $rows;
    }
}
